/*
 * Decompiled with CFR 0.152.
 */
package org.jabref.model.database;

import com.google.common.eventbus.EventBus;
import com.google.common.eventbus.Subscribe;
import java.math.BigInteger;
import java.security.SecureRandom;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.TreeSet;
import java.util.concurrent.ConcurrentHashMap;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import javafx.collections.FXCollections;
import javafx.collections.ObservableList;
import org.jabref.model.database.DuplicationChecker;
import org.jabref.model.database.KeyChangeListener;
import org.jabref.model.database.KeyCollisionException;
import org.jabref.model.database.event.EntryAddedEvent;
import org.jabref.model.database.event.EntryRemovedEvent;
import org.jabref.model.entry.BibEntry;
import org.jabref.model.entry.BibtexString;
import org.jabref.model.entry.InternalBibtexFields;
import org.jabref.model.entry.Month;
import org.jabref.model.entry.event.EntryEventSource;
import org.jabref.model.entry.event.FieldChangedEvent;
import org.jabref.model.strings.StringUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BibDatabase {
    private static final Logger LOGGER = LoggerFactory.getLogger(BibDatabase.class);
    private static final Pattern RESOLVE_CONTENT_PATTERN = Pattern.compile(".*#[^#]+#.*");
    private final ObservableList<BibEntry> entries = FXCollections.synchronizedObservableList((ObservableList)FXCollections.observableArrayList());
    private final Map<String, BibtexString> bibtexStrings = new ConcurrentHashMap<String, BibtexString>();
    private final DuplicationChecker duplicationChecker = new DuplicationChecker();
    private final Set<String> internalIDs = new HashSet<String>();
    private final EventBus eventBus = new EventBus();
    private String preamble;
    private String epilog = "";
    private String sharedDatabaseID;

    public BibDatabase() {
        this.eventBus.register(this.duplicationChecker);
        this.registerListener(new KeyChangeListener(this));
    }

    @Deprecated
    public static String getText(String toResolve, BibDatabase database) {
        if (toResolve != null && database != null) {
            return database.resolveForStrings(toResolve);
        }
        return toResolve;
    }

    public int getEntryCount() {
        return this.entries.size();
    }

    public boolean hasEntries() {
        return !this.entries.isEmpty();
    }

    public synchronized List<BibEntry> getEntriesSorted(Comparator<BibEntry> comparator) {
        ArrayList<BibEntry> entriesSorted = new ArrayList<BibEntry>((Collection<BibEntry>)this.entries);
        entriesSorted.sort(comparator);
        return entriesSorted;
    }

    public boolean containsEntryWithId(String id) {
        return this.internalIDs.contains(id);
    }

    public ObservableList<BibEntry> getEntries() {
        return FXCollections.unmodifiableObservableList(this.entries);
    }

    public Set<String> getAllVisibleFields() {
        TreeSet<String> allFields = new TreeSet<String>();
        for (BibEntry e : this.getEntries()) {
            allFields.addAll(e.getFieldNames());
        }
        return allFields.stream().filter(field2 -> !InternalBibtexFields.isInternalField(field2)).collect(Collectors.toSet());
    }

    public synchronized Optional<BibEntry> getEntryByKey(String key) {
        for (BibEntry entry : this.entries) {
            if (!key.equals(entry.getCiteKeyOptional().orElse(null))) continue;
            return Optional.of(entry);
        }
        return Optional.empty();
    }

    public synchronized List<BibEntry> getEntriesByKey(String key) {
        ArrayList<BibEntry> result = new ArrayList<BibEntry>();
        for (BibEntry entry : this.entries) {
            entry.getCiteKeyOptional().ifPresent(entryKey -> {
                if (key.equals(entryKey)) {
                    result.add(entry);
                }
            });
        }
        return result;
    }

    public synchronized Optional<BibEntry> getEntryById(String id) {
        return this.entries.stream().filter(entry -> entry.getId().equals(id)).findFirst();
    }

    public synchronized boolean insertEntry(BibEntry entry) throws KeyCollisionException {
        return this.insertEntry(entry, EntryEventSource.LOCAL);
    }

    public synchronized boolean insertEntry(BibEntry entry, EntryEventSource eventSource) throws KeyCollisionException {
        this.insertEntries(Collections.singletonList(entry), eventSource);
        return this.duplicationChecker.isDuplicateCiteKeyExisting(entry);
    }

    public synchronized void insertEntries(BibEntry ... entries) throws KeyCollisionException {
        this.insertEntries(Arrays.asList(entries), EntryEventSource.LOCAL);
    }

    public synchronized void insertEntries(List<BibEntry> entries) throws KeyCollisionException {
        this.insertEntries(entries, EntryEventSource.LOCAL);
    }

    private synchronized void insertEntries(List<BibEntry> newEntries, EntryEventSource eventSource) throws KeyCollisionException {
        Objects.requireNonNull(newEntries);
        for (BibEntry entry : newEntries) {
            String id = entry.getId();
            if (this.containsEntryWithId(id)) {
                throw new KeyCollisionException("ID is already in use, please choose another");
            }
            this.internalIDs.add(id);
            entry.registerListener(this);
            this.eventBus.post(new EntryAddedEvent(entry, eventSource));
        }
        this.entries.addAll(newEntries);
    }

    public synchronized void removeEntry(BibEntry toBeDeleted) {
        this.removeEntry(toBeDeleted, EntryEventSource.LOCAL);
    }

    public synchronized void removeEntry(BibEntry toBeDeleted, EntryEventSource eventSource) {
        Objects.requireNonNull(toBeDeleted);
        boolean anyRemoved = this.entries.removeIf(entry -> entry.getId().equals(toBeDeleted.getId()));
        if (anyRemoved) {
            this.internalIDs.remove(toBeDeleted.getId());
            this.eventBus.post(new EntryRemovedEvent(toBeDeleted, eventSource));
        }
    }

    public synchronized Optional<String> getPreamble() {
        if (StringUtil.isBlank(this.preamble)) {
            return Optional.empty();
        }
        return Optional.of(this.preamble);
    }

    public synchronized void setPreamble(String preamble) {
        this.preamble = preamble;
    }

    public synchronized void addString(BibtexString string) throws KeyCollisionException {
        if (this.hasStringLabel(string.getName())) {
            throw new KeyCollisionException("A string with that label already exists");
        }
        if (this.bibtexStrings.containsKey(string.getId())) {
            throw new KeyCollisionException("Duplicate BibTeX string id.");
        }
        this.bibtexStrings.put(string.getId(), string);
    }

    public void removeString(String id) {
        this.bibtexStrings.remove(id);
    }

    public Set<String> getStringKeySet() {
        return this.bibtexStrings.keySet();
    }

    public Collection<BibtexString> getStringValues() {
        return this.bibtexStrings.values();
    }

    public BibtexString getString(String id) {
        return this.bibtexStrings.get(id);
    }

    public Optional<BibtexString> getStringByName(String name) {
        for (BibtexString string : this.getStringValues()) {
            if (!string.getName().equals(name)) continue;
            return Optional.of(string);
        }
        return Optional.empty();
    }

    public int getStringCount() {
        return this.bibtexStrings.size();
    }

    public boolean hasNoStrings() {
        return this.bibtexStrings.isEmpty();
    }

    public void copyPreamble(BibDatabase database) {
        this.setPreamble(database.getPreamble().orElse(""));
    }

    public synchronized boolean hasStringLabel(String label) {
        for (BibtexString value : this.bibtexStrings.values()) {
            if (!value.getName().equals(label)) continue;
            return true;
        }
        return false;
    }

    public String resolveForStrings(String content) {
        Objects.requireNonNull(content, "Content for resolveForStrings must not be null.");
        return this.resolveContent(content, new HashSet<String>(), new HashSet<String>());
    }

    public Collection<BibtexString> getUsedStrings(Collection<BibEntry> entries) {
        ArrayList<BibtexString> result = new ArrayList<BibtexString>();
        HashSet<String> allUsedIds = new HashSet<String>();
        for (BibEntry entry : entries) {
            for (String fieldContent : entry.getFieldValues()) {
                this.resolveContent(fieldContent, new HashSet<String>(), allUsedIds);
            }
        }
        if (this.preamble != null) {
            this.resolveContent(this.preamble, new HashSet<String>(), allUsedIds);
        }
        for (String stringId : allUsedIds) {
            result.add((BibtexString)this.bibtexStrings.get(stringId).clone());
        }
        return result;
    }

    public List<BibEntry> resolveForStrings(Collection<BibEntry> entriesToResolve, boolean inPlace) {
        Objects.requireNonNull(entriesToResolve, "entries must not be null.");
        ArrayList<BibEntry> results = new ArrayList<BibEntry>(entriesToResolve.size());
        for (BibEntry entry : entriesToResolve) {
            results.add(this.resolveForStrings(entry, inPlace));
        }
        return results;
    }

    public BibEntry resolveForStrings(BibEntry entry, boolean inPlace) {
        BibEntry resultingEntry = inPlace ? entry : (BibEntry)entry.clone();
        for (Map.Entry<String, String> field2 : resultingEntry.getFieldMap().entrySet()) {
            resultingEntry.setField(field2.getKey(), this.resolveForStrings(field2.getValue()));
        }
        return resultingEntry;
    }

    private String resolveString(String label, Set<String> usedIds, Set<String> allUsedIds) {
        Objects.requireNonNull(label);
        Objects.requireNonNull(usedIds);
        Objects.requireNonNull(allUsedIds);
        for (BibtexString string : this.bibtexStrings.values()) {
            if (!string.getName().equalsIgnoreCase(label)) continue;
            if (usedIds.contains(string.getId())) {
                LOGGER.info("Stopped due to circular reference in strings: " + label);
                return label;
            }
            usedIds.add(string.getId());
            if (allUsedIds != null) {
                allUsedIds.add(string.getId());
            }
            String result = string.getContent();
            result = this.resolveContent(result, usedIds, allUsedIds);
            usedIds.remove(string.getId());
            return result;
        }
        Optional<Month> month = Month.getMonthByShortName(label);
        return month.map(Month::getFullName).orElse(null);
    }

    private String resolveContent(String result, Set<String> usedIds, Set<String> allUsedIds) {
        String res = result;
        if (RESOLVE_CONTENT_PATTERN.matcher(res).matches()) {
            int next;
            StringBuilder newRes = new StringBuilder();
            int piv = 0;
            while ((next = res.indexOf(35, piv)) >= 0) {
                int stringEnd;
                if (next > 0) {
                    newRes.append(res.substring(piv, next));
                }
                if ((stringEnd = res.indexOf(35, next + 1)) >= 0) {
                    String refLabel = res.substring(next + 1, stringEnd);
                    String resolved = this.resolveString(refLabel, usedIds, allUsedIds);
                    if (resolved == null) {
                        newRes.append(res.substring(next, stringEnd + 1));
                    } else {
                        newRes.append(resolved);
                    }
                    piv = stringEnd + 1;
                    continue;
                }
                newRes.append(res.substring(next));
                piv = res.length();
                break;
            }
            if (piv < res.length() - 1) {
                newRes.append(res.substring(piv));
            }
            res = newRes.toString();
        }
        return res;
    }

    public String getEpilog() {
        return this.epilog;
    }

    public void setEpilog(String epilog) {
        this.epilog = epilog;
    }

    public void registerListener(Object listener) {
        this.eventBus.register(listener);
    }

    public void unregisterListener(Object listener) {
        try {
            this.eventBus.unregister(listener);
        }
        catch (IllegalArgumentException e) {
            LOGGER.debug("Problem unregistering", e);
        }
    }

    @Subscribe
    private void relayEntryChangeEvent(FieldChangedEvent event) {
        this.eventBus.post(event);
    }

    public Optional<BibEntry> getReferencedEntry(BibEntry entry) {
        return entry.getField("crossref").flatMap(this::getEntryByKey);
    }

    public Optional<String> getSharedDatabaseID() {
        return Optional.ofNullable(this.sharedDatabaseID);
    }

    public void setSharedDatabaseID(String sharedDatabaseID) {
        this.sharedDatabaseID = sharedDatabaseID;
    }

    public boolean isShared() {
        return this.getSharedDatabaseID().isPresent();
    }

    public void clearSharedDatabaseID() {
        this.sharedDatabaseID = null;
    }

    public String generateSharedDatabaseID() {
        this.sharedDatabaseID = new BigInteger(128, new SecureRandom()).toString(32);
        return this.sharedDatabaseID;
    }

    public DuplicationChecker getDuplicationChecker() {
        return this.duplicationChecker;
    }
}

